/* File         :    scnparm.h

   Description  :    Scanner parameters header file.

   Targets      :    1394 Protocol.

   Conditions   :    None.

   Author       :    Loc Truong.

   Copyright    :    ( c ) 1999, Identix, Inc.

   Date         :    03.29.99    Created.
 */

#ifndef SCN_PARM__H
#define SCN_PARM__H

#include  "scntype.h"
#include  "imfparm.h"

/******************************/

/*
** Command Codes
**
** (An '*' in the comment means there is an associated data block structure.)
*/
#define CMD_GRB_TEST_MODE       36      // * Grabber Test Mode Control

#define CMD_SET_PARAMETERS      90      // * Set Parameters
#define CMD_GET_PARAMETERS      91      // * Get Parameters
#define CMD_SAVE_PARAMETERS     92      //   Save Parameters to EEPROM
#define CMD_RESTORE_PARAMETERS  93      //   Restore Parameters from EEPROM
#define CMD_CHECK_PARAMETERS    94      // * Check Parameters
#define CMD_RESTORE_DEF_PARAMS  95      //   Restore Parameters from 'C' Inits
#define CMD_GET_DEFAULT_PARAMETERS 96   // * Get default parameter values

#define CMD_UPDATE_FNGR_SENSOR  110     //   Update Finger Sensor Parameters


/*
** Grabber Test Modes
*/
typedef enum
{
    GRB_TEST_DISABLE,           // disable test mode
    GRB_TEST_ENABLE             // enable test mode
} Grb_TestMode;


/*
** Parameter Sets (ones with * are no longer active)
*/
typedef enum
{
    PRM_ASSY_MFG,       // scanner assembly manufacturing information
    PRM_FNGR_FINAL,     // final fingerprint specs for customer images
    PRM_FNGR_CONTRAST,  // final fingerprint contrast curve parameters
    PRM_FNGR_GRID,      // final fingerprint calibration grid points
    PRM_FNGR_CAPTURE,   // finger scanner capture pipeline parameters
    PRM_FNGR_EQUALIZE,  // finger scanner equalization parameters
    PRM_FNGR_EQU_TBL,   // finger scanner equalization table
    PRM_FNGR_FILTER,    // finger scanner reference image filter parameters
    PRM_FNGR_MINIMIZE,  // finger scanner minimized capture parameters
    PRM_FNGR_MIN_TBL,   // finger scanner minimized capture difference table
    PRM_FNGR_CONTACT,   // finger scanner contact parameters
    PRM_FNGR_EDGES,     // finger scanner edge detection parameters
    PRM_FNGR_SMEAR,     // finger scanner tip-smear attenuation parameters
    PRM_FNGR_LATENT,    // finger scanner latent removal parameters
    PRM_HAND_ROLL,      // hand scanner handling of declaring too fast
    PRM_DARKEN,         // Image darkening parameters
    PRM_FNGR_AUTOROLL_PREVIEW,  // finger scanner roll autocapture parameters
    PRM_FNGR_AUTOCAPT,  // finger scanner finger autocapture parameters
    PRM_FNGR_AUTOCAPT_NOPREVIEW,// finger scanner finger ink mode autocapture parameters
    PRM_SLPP_MIN_TBL,   //*slap scanner preview minimized capture diff table
    PRM_SLPP_CONTACT,   //*slap scanner preview contact parameters
    PRM_SLPP_EDGES,     //*slap scanner preview edge detection parameters
    PRM_SLPS_FINAL,     // final slap print specs for customer images
    PRM_SLPS_CONTRAST,  // final slap print contrast curve parameters
    PRM_SLPS_GRID,      // final slap print calibration grid points
    PRM_SLPS_CAPTURE,   // slap scanner scan capture parameters
    PRM_SLPS_EQUALIZE,  // slap scanner scan equalization parameters
    PRM_SLPS_EQU_TBL,   // slap scanner scan equalization table
    PRM_SLPS_FILTER,    // slap scanner scan reference image filter parameters
    PRM_SLPS_MINIMIZE,  // slap scanner scan minimized capture parameters
    PRM_SLPS_MIN_TBL,   // slap scanner scan minimized capture difference table
    PRM_SLPS_CONTACT,   // slap scanner scan contact parameters
    PRM_SLPS_EDGES,     // slap scanner scan edge detection parameters
    PRM_SLPS_LATENT,    // slap scanner scan latent removal parameters
    PRM_SLAP_CONFIG,    // slap scanner hardware configuration parameters
    PRM_FNGR_RTQA,      // finger scanner real time quality assurance params
    PRM_FNGR_CHECK,     // finger scanner platen check parameters
    PRM_SLPP_CHECK,     //*slap scanner preview platen check parameters
    PRM_SLPS_CHECK,     // slap scanner scan platen check parameters
    PRM_FNGR_SHARPEN,   // finger image sharpening parameters
    PRM_SLPS_SHARPEN,   // slap image sharpening parameters
    PRM_FNGR_MFG,       // finger scanner unit manufacturing information
    PRM_SLAP_MFG,       // slap scanner unit manufacturing information
    PRM_FNGR_CONFIG,    // finger scanner hardware configuration parameters
    PRM_FNGR_SENSOR,    // slap scanner sensor values
    PRM_HAND_MFG,       // hand scanner unit manufacturing information
    PRM_HAND_CONFIG,    // hand scanner hardware configuration parameters
    PRM_HAND_FINAL,     // final handprint specs for customer images
    PRM_HAND_CONTRAST,  // final handprint contrast curve parameters
    PRM_HAND_SEGS,      // final handprint calibration segment points
    PRM_HAND_CAPTURE,   // hand scanner capture parameters
    PRM_HAND_EQUALIZE,  // hand scanner equalization parameters
    PRM_HAND_EQU_TBL,   // hand scanner equalization table
    PRM_HAND_FILTER,    // hand scanner reference image filter parameters
    PRM_HAND_MINIMIZE,  // hand scanner minimized capture parameters
    PRM_HAND_MIN_TBL,   // hand scanner minimized capture difference table
    PRM_HAND_CONTACT,   // hand scanner contact parameters
    PRM_HAND_CHECK,     // hand scanner roller check parameters
    PRM_HAND_SHARPEN,   // hand image sharpening parameters
    PRM_HAND_SPEED,     // hand speed indicator parameters
    PRM_HAND_EDGES,     // hand scanner edge detection parameters
    PRM_COMBO_MFG,      // combo scanner unit manufacturing information
    PRM_COMBO_CONFIG,   // combo scanner hardware configuration parameters
    PRM_CAMERA_CONFIG,  // camera configuration parameters (not really a parameter set)
    PRM_CAPT_THRESHOLD, // Guardian capture threshold values
    PRM_FNGR_BREAKS,    // finger scanner ridge break detection threshold
    PRM_FNGR_CCA,       // finger scanner cellular contrast adjustment
    PRM_SLPS_RTQA,      // slap scanner real time quality assurance params
    PRM_SLPS_CCA,       // slap scanner cellular contrast adjustment
    PRM_HAND_RTQA,      // hand scanner real time quality assurance params
    PRM_HAND_CCA,       // hand scanner cellular contrast adjustment
    PRM_FNGR_CBC,       // finger scanner cellular background cleaning
    PRM_SLPS_CBC,       // slap scanner cellular background cleaning
    PRM_HAND_CBC,       // hand scanner cellular background cleaning
    PRM_AUTO_SLAP,      // Guardian auto-detection values
    PRM_FNGR_THRESHOLDS_SLAP,// Individual thresholds for all fingers
    PRM_AUTOCAPT_TIMERS_SLAP,// programmable timeouts for auto-capture
    PRM_DETECT_SLAP_TYPE,//thresholds detecting left, right, or center two
    PRM_ADAPTIVE_FILTER, // values for the adaptive filter in MoreRes.c 
    PRM_SLAP_FNGRGAP,    // values for the determining multiple finger
    PRM_RT_RTQA,         // determines threshold for real time RTQA 
    PRM_PALM_FINAL,     // final palmprint specs for customer images
    PRM_PALM_CONTRAST,  // final palmprint contrast curve parameters
    PRM_PALM_GRID,      // final palmprint calibration grid points
    PRM_PALM_CAPTURE,   // palm scanner capture pipeline parameters
    PRM_PALM_EQUALIZE,  // palm scanner equalization parameters
    PRM_PALM_CONTACT,   // palm scanner contact parameters
    PRM_PALM_LATENT,    // palm scanner latent removal parameters
    PRM_PALM_SHARPEN,   // palm image sharpening parameters
    PRM_PALM_CCA,       // palm scanner cellular contrast adjustment
    PRM_PALM_CBC,       // palm scanner cellular background cleaning
    PRM_PALM_RTQA,      // palm scanner real time quality assurance params
    PRM_FNGR_DISP,      // configuration of the display pallet
    PRM_SLPS_DISP,      // configuration of the display pallet
    PRM_PALM_DISP,      // configuration of the display pallet
    PRM_COUNT            // number of defined parameter sets
} Parameter_Set;


/****************************/
//
// Parameter data types
//

#define NO_OF_PRM_RT_RTQA 10

typedef struct        // for PRM_RT_RTQA
//
//
{
    U32         sGoodPct[NO_OF_PRM_RT_RTQA]; // percentage value for determining 
                              // whether the quality is good 
} Prm_RtRtqa;

typedef struct        // for PRM_SLAP_FNGRGAP
//
//
{
    U32         uHeight;    // height of gap, 0 disables
    U32         uWidth;     // width of gap
    U32         uLinesSkipped;// no. of contact lines skipped at the top of
                            // the image
    U32         uCenterChg; // size of minimun center line detection
} Prm_SlapFngrGap;

typedef struct        // for PRM_ASSY_MFG
//
// The version number, serial number and date of manufacture or assembly
// are noted here.
//
{
    char        make[8];    // e.g. "IDX"
    char        model[16];  // e.g. "TP-2000"
    char        serial[16]; // e.g. "123456"
    U32         mfgYear;    // e.g. 2001
    U32         mfgMonth;   // 1-12
    U32         mfgDay;     // 1-31
    U32         mfgScanners;// assembled scanner units
} Prm_AssemblyMfg;

typedef enum
//
// Final contrast curves may be formed by linearly stretching
// the raw contrast curve to extend to the designated black
// and white limits, or by flattening the final contrast
// curve.  The flattening technique is referred to here
// as "integrated" since it is implemented via an analysis
// of the integrated histogram.
//
{
    cont_linear,
    cont_integrated,
    cont_none
} Contrast_Method;

typedef enum
//
// Sharpening can be done with one of four methods.  The
// "full" method takes into account the point itself, and
// its eight nearest neighbors.  The "quad" method takes 
// into account the point itself, and the four points above,
// beneath, left and right.  The "vertical" method, which
// is specifically intended for use with the slap scanner,
// takes into account the point itself and the two above
// and beneath it.  The "h_and_v" method uses the normal
// center weight to filter in the horizontal domain, and
// a second (alternate) center weight to filter in the
// vertical domain; this method has the same cross shape
// as for the "quad" method.
//
{
    sharpen_quad,
    sharpen_full,
    sharpen_vertical,
    sharpen_horizontal, //Not used in the Jaguar
    sharpen_ortho       //Not used in the Jaguar
} Sharpen_Method;

typedef struct          // for PRM_FNGR_CAPTURE and PRM_SLPS_CAPTURE
//
// Capture Parameters
//
// The sum of "leading", and "height" must be less than or equal to the
// maximum number of scan lines (as defined by the xxxx_MAX_LINES constant
// for each scanner type).  Likewise, the sum of "inset" and "width" must
// be less than or equal to the maximum number of pixels on a scan line
// (as defined by the xxxx_MAX_PIXELS constant for each scanner type).
//
// Moreover, for the finger scanner "leading" must be no less than two.
// 
{
    U32 leading;        // scan lines to discard at top-of-frame
    U32 inset;          // pixels to discard at left end of each scan line
    U32 height;         // scan lines retained from each frame
    U32 width;          // pixels retained from each scan line
} Prm_Capture;


typedef struct          // for PRM_FNGR_EQU_TBL, PRM_SLPS_EQU_TBL,
                        // and PRM_HAND_EQU_TBL
//
// Equalization Table
//
{
    U8 table[131072];   // loaded equalization table
} Prm_EqualizeTable;

typedef struct          // for PRM_FNGR_FILTER and PRM_SLPS_FILTER
//
// Equalization Reference Image Processing Parameters
//
// These parameters are used by the calibration commands to process the
// captured image into a reference image.  If the calibration command does
// not require this processing then the host must load a reference image.
// The reference image is always the maximum size possible from the scanner.
//
// Reference image processing forces all border areas to black.  The inside
// is smoothed using a rectangular filter.  Each pixel on which the filter
// can be centered, without overlapping a border area, is assigned the mean
// level of all pixels covered by the filter.  Pixels that are too near the
// borders for the filter to be centered on them are assigned the value of
// the nearest pixel that can be filtered.
{
    U32 top;            // scan lines in top border
    U32 left;           // pixels in left border
    U32 right;          // pixels in right border
    U32 bottom;         // scan lines in bottom border
    U32 width;          // width (in pixels) of filter (must be odd)
    U32 height;         // height (in scan lines) of filter (must be odd)
} Prm_RefFilter;

typedef struct          // for PRM_FNGR_MINIMIZE, PRM_SLPS_MINIMIZE
                        // and PRM_HAND_MINIMIZE
//
// Minimized Capture Parameters
//
{
    U32 loaded;         // == 0 -> generate table,  != 0 -> use loaded table
    U32 attack;         // 1-100: attack slope percentage (for generated table)
} Prm_Minimize;

typedef struct          // for PRM_FNGR_MIN_TBL, PRM_SLPS_MIN_TBL
                        // and PRM_HAND_MIN_TBL
//
// Minimized Capture Difference Table
//
{
    U8 table[256];      // loaded minimized capture table
} Prm_MinimizeTable;

typedef struct          // for PRM_FNGR_CONTACT
//
// Finger Scanner Contact Parameters
//
{
    U32 threshold;      // grayscale at or below which contact is assumed
    U32 motion;         // average horizontal movement needed to report motion
} Prm_FngrContact;

typedef struct          // for PRM_FNGR_EDGES, PRM_SLPS_EDGES,
                        // and PRM_HAND_EDGES
//
// Edge Detection Parameters
//
{
    U32 detection;      // contact pixels needed for edge detection
    U32 expansion;      // pixels by which edges are expanded before use
    U32 filter;         // 0-15: initial edge filter half-interval
    U32 verticality;    // number of lines needed for frame contact
} Prm_Edges;

typedef struct          // for PRM_FNGR_SMEAR
//
// Finger Scanner Tip-Smear Attenuation Parameters
//
// Extra edge smoothing is done with the following forumula
//
//      E[l] = (F * e) + ((1 - F) * E[l-1])
//
// where:
//           E[l]    is smoothed edge for current line
//           E[l-1]  is smoothed edge for preceding line
//           e       is measured edge for current line
//           F       is smoothing factor (0 < F <= 1)
//
// The smoothing factor, F, is determined from the smoothing parameter as
// follows
//
//      F = round(smoothing / 100)
//
// From the above, it can be seen that the maximum allowed smoothing parameter
// (100) does no smoothing at all, and lesser smoothing parameters smooth more
// aggressively.
//
// The middle line is the vertical position of the foremost leading edge of
// contact.  As a roll progresses, this position may move up or down.  This
// vertical motion is also smoothed using the same method as above.
//
// As a roll progresses, the freeze-point on a scan line generally moves in
// the direction of the roll.  This freeze-point is constrained from backward
// motion, and its forward motion is smoothed using the same method as above.
//
{
    U32 freeze;         // 0-100: freeze-point position percentage
                        //             (from trailing to leading edge)
    U32 freeze_hi;      // 0-100: upper freeze-point limit percentage
    U32 freeze_lo;      // 0-100: lower freeze-point limit percentage
    U32 smooth_edges;   // 1-100: line-to-line extra edge smoothing percentage
    U32 smooth_midline; // 1-100: middle line movement smoothing percentage
    U32 smooth_freeze;  // 1-100: line-to-line freeze-point smoothing percent
    U32 monitor;        // 0-N: monitor roll for change in motion
} Prm_FngrSmear;


typedef struct          // for PRM_SLPS_CONTACT and PRM_HAND_CONTACT
//
// Slap Scanner Contact Parameters
//
{
    U32 threshold;      // grayscale at or below which contact is assumed
} Prm_Contact;

typedef struct          // for PRM_SLAP_CONFIG
{
//
// Slap Scanner Hardware Configuration Parameters
//
    U32 preview;        // 0 - no preview camera, !0 - preview camera present
    U32 preview_light;  // illumination level for preview mode
    U32 scan_light;     // illumination level for scan mode
    U32 scan_gain;      // gain setting for scan mode
} Prm_SlapConfig;

typedef struct          // for PRM_FNGR_RTQA
{
//
// Real Time Quality Assurance Parameters for fingers.
//
// The percentile parameters are used to establish dark and
// light values for the image as a whole.  The dark value is
// chosen to be that value for which dark_percentile of the
// rtqa cells have darker minima.  The light value is chosen
// to be that value for which light_percentile of the rtqa cells 
// have darker maxima.  
//
// Once the percentiles have been used to select light and 
// dark values, these values are subjected to a limit clip.
// The chosen dark value may be no greater than max_dark.  The
// chosen light value may be no less than min_light.
//
// Once the light and dark values are chosen, they are used
// to form the criteria for judging individual cells.
//
// The threshold values chosen are based on the light-dark range.
// Specifically, a cell will be considered to have detectable
// print data only if its darkest value is less than the
// detect_print percentage point in that range.  If the lightest
// value in a cell is less than the too_dark percentage point
// in that range, then the cell is deemed to be too dark.  If
// the darkest value in a cell is greater than the too_light
// percentage point in that range, then the cell is deemed to be
// too light.  If the difference between the lightest and darkest
// values in a cell is less than too_blurred times the light-dark
// range, then that cell is deemed to be blurred (smeared).
//
    U32 cell_width;       // width of analysis cell
    U32 cell_height;      // height of analysis cell
    U32 data_smooth;      // crease finder (data smoothing interval)
    U32 slope_smooth;     // crease finder (slope smoothing interval)
    U32 dark_percentile;  // percentile used in determining range
    U32 light_percentile; // percentile used in determining range
    U32 max_dark;         // max allowed dark value
    U32 min_light;        // min allowed light value

    U32 detect_print;     // percentage of light-dark range
    U32 too_dark;         // percentage of light-dark range
    U32 too_light;        // percentage of light-dark range
    U32 too_blurred;      // percentage of light-dark range
} Prm_FngrRtqa;

typedef struct          // for PRM_SLPS_RTQA and PRM_HAND_RTQA
{
//
// Real Time Quality Assurance Parameters for Slap-capture.
//
// This is similar to the RTQA parameters for fingers, but
// without those related to finding the crease.
//
    U32 cell_width;       // width of analysis cell
    U32 cell_height;      // height of analysis cell
    U32 dark_percentile;  // percentile used in determining range
    U32 light_percentile; // percentile used in determining range
    U32 max_dark;         // max allowed dark value
    U32 min_light;        // min allowed light value

    U32 detect_print;     // percentage of light-dark range
    U32 too_dark;         // percentage of light-dark range
    U32 too_light;        // percentage of light-dark range
    U32 too_blurred;      // percentage of light-dark range
} Prm_Rtqa;

typedef struct        // for PRM_FNGR_CCA, PRM_SLPS_CCA, and PRM_HAND_CCA
//
// Cellular Contrast Adjustment
//
// This parameter set controls cellular contrast adjustment processing.  The
// value of the maximum stretch parameter is 100 times the maximum allowed
// factor used in the linear mapping of the pixel values in a cell.  The
// minimum squeeze parameter is 100 times the minimum allowed factor.
//
// The other parameters are similar to those specified for RTQA.  They control
// the determination of the representative dark and light levels for CCA.  The
// region limit parameter acts like the crease does for RTQA.  No part of the
// fingerprint below this line will be use for determining the representative
// levels.  However, CCA is applied to all cells in the fingerprint, even
// those below this limit (as well as those below the actual crease).
//
{
    U32 region_limit;     // last line used for dark/light determination
    U32 dark_percentile;  // percentile used in determining range
    U32 light_percentile; // percentile used in determining range
    U32 max_dark;         // maximum allowed dark value
    U32 min_light;        // minimum allowed light value
    U32 detect_print;     // percentage of light-dark range
    U32 bgnd_threshold;   // below background threshold for print detect (x10)
    U32 island_size;      // threshold for island of same type cells
    U32 bulge_size;       // threshold for bulge of same type cells at edge
    U32 max_stretch;      // maximum adjustment factor (times 100)
} Prm_CellAdjust;

typedef struct            // for PRM_FNGR_CHECK, PRM_SLPS_CHECK, 
{                         // and PRM_HAND_CHECK
//
// Platen Check Parameters
//
    U32 line_sampling;    // vertical sampling frequency
    U32 pixel_sampling;   // horizontal sampling frequency
    U32 dirty_threshold;  // illumination level for preview mode
    U32 damage_threshold; // illumination level for preview mode
} Prm_PlatenCheck;

typedef struct        // for PRM_FNGR_MFG, PRM_SLAP_MFG, and PRM_HAND_MFG
//
// The version number, serial number and date of manufacture or assembly
// are noted here.
//
{
    U32         version;    // 0-255
    U32         serial;     // 0-65535
    U32         mfgYear;    // e.g. 2001
    U32         mfgMonth;   // 1-12
    U32         mfgDay;     // 1-31
} Prm_ScannerMfg;

typedef struct        // for PRM_FNGR_CONFIG
//
// Finger Scanner Hardware Configuration Parameters
//
{
    U32         illuminator;    // 0-1023
    U32         clock_mode;     // 0-7
    U32         cds_mode;       // 0:disabled, 1:enabled
    U32         clamp_mode;     // 0:disabled, 1:enabled
    U32         vga_gain;       // 0-1023
    U32         cds_gain;       // 0-63
    U32         clamp_level;    // 0-255
} Prm_FngrConfig;

typedef struct        // for PRM_FNGR_SENSOR
//
// Finger Scanner Brightness Sensor and Other A/D Channels
//
{
    U32         brightness;
    U32         channel_1;
    U32         channel_3;
} Prm_FngrSensor;

typedef struct          // for PRM_HAND_CONFIG
{
//
// Hand Scanner Hardware Configuration Parameters
//
    U32 light;          // illumination level
    U32 gain;           // gain setting
} Prm_HandConfig;

typedef struct          // for below
//
// Each point has platen space coordinates and capture space
// coordinates.
//
// The drum space coordinates are in units of .001 inches.  By
// definition, the left endpoint of the calibration line is zero
// in drum space.  The name "drum space" is meant to imply that
// these are real measurements, taken on the surface of the drum.
//
// The capture space coordinates are in pixels.  In the process
// of calibration, the target is placed on the drum, and a raw
// image is captured.  The raw image is then viewed, and the
// pixel coordinate is determined for each calibration point.
//
{
    U32 px;             // platen coordinates in .001 in units
    U32 cx;             // capture coordinates in pixels
} Prm_SegPt;

#define MAX_CAL_SEG_PTS     16

typedef struct          // for PRM_HAND_SEGS
//
// The number of segment points used in calibration may
// vary.  These points are the endpoints of calibration
// segments.  Adjacent segments share endpoints, so ten
// points are required for a nine segment calibration.
//
// The leftmost endpoint is given first, then the one to
// its right, and so on.
//
// The hand segment array is annotated with a calibration
// date.
//
{
    U32         calYear;    // e.g. 2000
    U32         calMonth;   // e.g. 5
    U32         calDay;     // e.g. 20
    U32         calVRes;    // inherent vert resolution (dpi)
    U32         segPts;     // e.g. 10 for 9 segment cal
    Prm_SegPt   pt [MAX_CAL_SEG_PTS];  // point array
} Prm_CalibSeg;

typedef struct          // for PRM_HAND_CAPTURE
//
// Line Capture Parameters
//
// The sum of "inset" and "width" must be less than or equal to the maximum
// number of pixels on a scan line (as defined by the xxxx_MAX_PIXELS constant
// for each scanner type).
// 
{
    U32 inset;          // pixels to discard at left end of each scan line
    U32 width;          // pixels retained from each scan line
    U32 max_height;     // maximum lines stored when transforming lines
    U32 raw_height;     // maximum lines stored when not transforming lines
} Prm_HandCapt;

typedef struct          // for PRM_HAND_FILTER
//
// Equalization Reference Line Processing Parameters
//
// These parameters are used by the calibration commands to process the
// captured line into a reference line.  If the calibration command does
// not require this processing then the host must load a reference line.
// The reference line is always the maximum size possible from the scanner.
//
// Reference line processing forces all border areas to black.  The inside
// is smoothed using a fixed segment filter.  Each pixel on which the filter
// can be centered, without overlapping a border area, is assigned the mean
// level of all pixels covered by the filter.  Pixels that are too near the
// borders for the filter to be centered on them are assigned the value of
// the nearest pixel that can be filtered.
{
    U32 left;           // pixels in left border
    U32 right;          // pixels in right border
    U32 width;          // width (in pixels) of filter (must be odd)
} Prm_RefLnFilter;

typedef struct          // for PRM_HAND_EQUALIZE
//
// Tri-Linear Compression Equalization Parameters
//
{
    U32 loaded;         // == 0 -> generate table,  != 0 -> use loaded table
    U32 base;           // base offset level (for generated table)
    U32 peak;           // peak normalization level (for generated table)
    U32 w_cmp;          // white (bgnd) compression level (for generated table)
    U32 w_out;          // white (bgnd) compressed output (for generated table)
    U32 white;          // equalized white (bgnd) level (for generated table)
    U32 b_cmp;          // black (dark) compression level (for generated table)
    U32 b_out;          // black (dark) compressed output (for generated table)
    U32 black;          // equalized black (dark) level (for generated table)
} Prm_CompressEqu;

typedef struct          // for PRM_HAND_SPEED
//
// Hand Speed Indicator Parameters
//
// These parameters set the levels used with the hand speed indicator
// overlay feature.  The valid range for the number of levels is 0-10.
// Zero disables the feature.  If there is only one level specified
// then all lines will be assgned level zero, regardless of speed.
// Otherwise, there must be a specified speed for each level except
// the last one, and lines will be assigned levels as described below.
//
// Level zero will be assigned for any speed less than or equal to the
// slowest specified speed.  Level one will be assigned for any speed
// greater than the slowest specified speed and less than or equal to
// the second slowest specified speed.  This continues until the level
// that is one less than the specified number of levels (levels - 1)
// will be assigned for any speed greater than the fastest specified
// speed.
//
// All specifed speeds must be distinct and in the range 1-507.
{
    U32 levels;         // number of levels (0-10)
    U32 speed[9];       // up to nine speeds for max. number of levels
} Prm_HandSpeed;

typedef struct        // for PRM_COMBO_CONFIG
//
// Combo Scanner Hardware Configuration Parameters
//
{
    U32         test_pattern;   // 0-5
} Prm_ComboConfig;

typedef struct        // for PRM_CAMERA_CONFIG
//
// Camera Configuration Parameters
//
{
    U32 exposure1;
    U32 gain_a1;
    U32 gain_b1;
    U32 offset_a1;
    U32 offset_b1;
    U32 exposure2;
    U32 gain_a2;
    U32 gain_b2;
    U32 offset_a2;
    U32 offset_b2;
} Prm_CameraConfig;

typedef struct         // for PRM_CAPT_THRESHOLD
//
// Capture threshold parameters
//
{
    U32     uiRedThreshold;          // Threshold for minimum contact detection.
    U32     uiYellowThreshold;       // Threshold for intermediate contact detection.
    U32     uiGreenThreshold;        // Threshold for acceptble finger contact detected.
    U32     uiTimeOut;               // 
}Prm_CaptThreshold;

typedef struct         // for PRM_DARKEN
//
// Parameters for image darkening
//
{
    U32     uiBlackLevel;      // The start line 
    U32     uiTargetLevel;     // The end line
    U32     uiMinGoodLevel ;   // Minimum acceptable good level
    U32     uiMaxChange;       // Maximum alteration value
}Prm_Darken;


typedef struct        // for PRM_FNGR_BREAKS
//
// Rolled Fingerprint Ridge Break Detection
//
// This parameter set specifies the threshold for the ratio of an analysis
// measure across a freeze curve to that measure away from the freeze curve.
// Higher ratios indicate the possible presence of ridge breaks.  Ridge breaks
// occur when the subject finger slips during the roll.
//
// When this occurs at the tip of a finger (tip-smear), the ridge breaks at
// a freeze curve extend only part way into the fingerprint area.  When this
// occurs as a result of the entire finger sliding, the discontinuities can
// extend all the way from the top to the bottom of the fingerprint.
//
// This feature counts the number freeze curves where discontinuites appear
// along a significant portion of the entire curve.  These indicate sliding
// (or slippage) of the whole finger, and indicated a separate problem from
// other RTQA results.
//
// The value of the threshold parameter is the chosen ratio times 100.  The
// limit parameter is the number of wholly bad freeze curves allowed.  When
// set to zero this will indicate a bad capture if any wholly bad freeze
// curves occur.
//
// This feature analyzes freeze curves piece-wise.  The size of a freeze
// curve segment is specified as a number of lines in the image.  As the
// analysis proceeds down the freeze curve from the topmost line that is in
// the fingerprint area, it overlaps successive segments.  The amount of
// overlap is also specified as a number of lines.
//
{
    U32 break_ratio;      // ridge break ratio threshold (times 100)
    U32 break_limit;      // limit on no. of full height ridge breaks allowed
    U32 seg_size;         // lines in a freeze curve segment
    U32 seg_overlap;      // lines of overlap in two successive segments
    U32 cover_gap;        // lines in largest allowed coverage gap
    U32 cover_mean;       // threshold for mean coverage
    U32 blur_slip;        // non-zero means blur slippage detected cells
} Prm_FngrBreaks;

typedef struct        // for PRM_AUTO_SLAP
//
//
{
  U32   uiLowestThreshold;  // Threshold for minimum contact detection.
  U32   uiMiddleThreshold;  // Threshold for intermediate contact detection.
  U32   uiHighestThreshold; // Threshold for acceptable finger contact detected.
  U32   uiChangeThresholds; // Seconds before decreasing threshold values after contact
  U32   uiMaxCaptureDelay;  // Seconds before forcing capture after contact
  U32   uiCaptureDelay;     // Seconds delayed before capture after capture criteria met
  U32   uiLatentWait;       // Seconds in latent removal loop waiting for finger removal
  U32   uiThresholdAdj;     // Relative amount of threshold change per second 
} Prm_AutoSlap;

typedef struct         // for PRM_FNGR_AUTOROLL_PREVIEW
//
// Parameters for automatic finger roll capture
//
{
   U32   uiStartDelay;       // Seconds delayed before capture after capture criteria met
   U32   uiCaptureDelay;     // Seconds delayed before capture after capture criteria met
   U32   uiMaxCaptureDelay;  // Seconds before forcing capture after contact
   U32   uiLatentWait;       // Seconds in latent removal loop waiting for finger removal
   U32   uiChangeThresholds; // Seconds before decreasing threshold values after contact
   U32   uiLowestThreshold;  // Threshold for minimum contact detection.
   U32   uiHighestThreshold; // Threshold for acceptable finger contact detected.
   U32   uiThresholdAdj;     // Relative amount of threshold change per second 
   U32   uiCenterWidthPct;   // Size of center exclusion area used for auto-capture rolls
}Prm_AutoCapt;

typedef struct       //for PRM_HAND_ROLL
//
{
    U32  uiLength;         // accumulated vertical error in 100ths of an inch
    U32  uiNotUsed1;    
    U32  uiNotUsed2;
} Prm_RollSpeed;

#define NO_PRM_FINGERS     10
#define NO_PRM_THRESHOLDS  3
typedef struct        // for PRM_FNGR_THRESHOLDS

/* Three levels of threshold: The lowest determines how much contact is required to declare
 finger present. The middle threshold is to show contact greater than the minimum but still
 less than the required. The highest threshold is set to the level required to ensure good 
 finger contact.
 Each finger has a threshold setting:  The values for the three levels of threshold are 
 adjustable for each finger. */
{
   U32 ucNoOfPrimaryFingers;
   U32 usFingerThreshold[NO_PRM_FINGERS][NO_PRM_THRESHOLDS];  
}Prm_FngrThreshold; 

typedef struct        // for PRM_AUTOCAPT_TIMERS
/*
  There are four programmable delays.
   Hesitation after finger detection before updating the LED               
   Seconds delayed before capture after capture criteria met
   Number of seconds delayed before capture after alternate capture criteria has been met.
   Seconds before forcing capture after contact
*/
{
   U32   usCaptureDelay;     	   // Seconds delayed before capture after capture criteria met
   U32   usAltCaptureDelay;     // This time is associated with the number of primary fingers variable. Contains the number of seconds delayed before capture after alternate capture criteria has been met.
   U32   usMaxCaptureDelay;  	   // Seconds before forcing capture after contact
   U32   usLatentWait;           // Seconds in latent removal loop waiting for finger removal
}Prm_AutoCaptTimers; 

typedef struct        // for PRM_DETECT_SLAP_TYPE
/*
*/
{
   U32   ucOverlapThres;     	// pixel change with over lapped images
   U32   ucSeparateThres;    	// pixel change with independent image
   U32   usMinimumDiff;     //The minimum number of pixels difference required
                              // to determine slap type
}Prm_DetectSlapType; 


/****************************/

/*
** Status/Event Codes
**
** (An '*' in the comment means there is an associated data block structure.)
*/
#define STA_CHECK_PARAMETERS   150       //   Check Parameters Response


/*
** Basic State Codes
**
** The following define codes assigned to the basic state member for various
** status/event codes.
*/

typedef enum        // for STA_CHECK_PARAMETERS
{
    PARAM_VALID     = 0x01,     // parameter set and size is valid
    PARAM_SAVEABLE  = 0x02,     // parameter set has non-volatile storage
    PARAM_SAVED     = 0x04      // parameter set same as non-volatile storage
} Sta_CheckParam;

#endif
